definition module StdIOCommon


//	********************************************************************************
//	Clean Standard Object I/O library, version 1.1
//	
//	StdIOCommon defines common types and access functions for the 
//	Object I/O library.
//	********************************************************************************


import	StdOverloaded
import	StdString
import	StdIOBasic
from	id		import	Id, RId, R2Id, RIdtoId, R2IdtoId, toString, ==
from	key		import	SpecialKey, 
							BeginKey, 
							ClearKey, 
							DeleteKey, DownKey, 
							EndKey, EnterKey, EscapeKey, 
							F1Key,  F2Key,  F3Key,  F4Key,  F5Key,  
							F6Key, F7Key, F8Key, F9Key, F10Key,
							F11Key, F12Key, F13Key, F14Key, F15Key, 
							HelpKey, 
							LeftKey, 
							PgDownKey, PgUpKey, 
							RightKey, 
							UpKey

/*	General type constructors for composing local and context-dependent 
	data structures.
*/
::	SelectState		=	Able | Unable
::	MarkState		=	Mark | NoMark

enabled		:: !SelectState	-> Bool			// @1 == Able
marked		:: !MarkState	-> Bool			// @1 == Mark

instance	== SelectState					// Constructor equality
instance	== MarkState					// Constructor equality
instance	~  SelectState					// Able <-> Unable
instance	~  MarkState					// Mark <-> NoMark

::	KeyboardState
	=	CharKey		Char		KeyState	// ASCII character input
	|	SpecialKey	SpecialKey	KeyState Modifiers
											// Special key input
::	KeyState
	=	KeyDown		IsRepeatKey				// Key is down
	|	KeyUp								// Key goes up
::	IsRepeatKey								// Flag on key down:
	:==	Bool								// True iff key is repeating
::	Key
	=	IsCharKey	 Char
	|	IsSpecialKey SpecialKey
::	KeyboardStateFilter						// Predicate on KeyboardState:
	:==	KeyboardState -> Bool				// evaluate KeyFunction only if True

getKeyboardStateKeyState:: !KeyboardState -> KeyState
getKeyboardStateKey		:: !KeyboardState -> Key

instance	== KeyState						// Equality on KeyState


::	MouseState
	=	MouseMove	Point Modifiers			// Mouse is up     (position,modifiers)
	|	MouseDown	Point Modifiers Int		// Mouse goes down (and nr down)
	|	MouseDrag	Point Modifiers			// Mouse is down   (position,modifiers)
	|	MouseUp		Point Modifiers			// Mouse goes up   (position,modifiers)
::	ButtonState
 	=	ButtonStillUp						// MouseMove
 	|	ButtonDown							// MouseDown _ _ 1
	|	ButtonDoubleDown					//			 _ _ 2
	|	ButtonTripleDown					//           _ _ >2
	|	ButtonStillDown						// MouseDrag
 	|	ButtonUp							// MouseUp
::	MouseStateFilter						// Predicate on MouseState:
	:==	MouseState -> Bool					// evaluate MouseFunction only if True

getMouseStatePos		:: !MouseState	-> Point
getMouseStateModifiers	:: !MouseState	-> Modifiers
getMouseStateButtonState:: !MouseState	-> ButtonState

instance	== ButtonState					// Constructor equality


::	SliderState
	=	{	sliderMin	:: !Int
		,	sliderMax	:: !Int
		,	sliderThumb	:: !Int
		}

instance == SliderState						// @1.sliderMin   == @2.sliderMin
											// @1.sliderMax   == @2.sliderMax
											// @1.sliderThumb == @2.sliderThumb


::	UpdateState
	=	{	oldFrame	:: !ViewFrame
		,	newFrame	:: !ViewFrame
		,	updArea		:: !UpdateArea
		}
::	ViewDomain			:==	Rectangle
::	ViewFrame			:==	Rectangle
::	UpdateArea			:==	[ViewFrame]



::	Modifiers
	=	{	shiftDown	:: !Bool			// True iff shift   down
		,	optionDown	:: !Bool			// True iff option  down
		,	commandDown	:: !Bool			// True iff command down
		,	controlDown	:: !Bool			// True iff control down
		,	altDown		:: !Bool			// True iff alt     down
		}

//	Constants to check which of the Modifiers are down.

NoModifiers	:==	{shiftDown	= False
				,optionDown	= False
				,commandDown= False
				,controlDown= False
				,altDown	= False
				}
ShiftOnly	:==	{NoModifiers & shiftDown	= True}
OptionOnly	:== {NoModifiers & optionDown	= True}
CommandOnly	:== {NoModifiers & commandDown	= True}
ControlOnly	:== {NoModifiers & controlDown	= True}
AltOnly		:==	{NoModifiers & altDown		= True}


/*	The layout language used for windows and controls.
*/
::	ItemPos
	:==	(	ItemLoc
		,	ItemOffset
		)
::	ItemLoc
	//	Absolute:
	=	Fix		Point
	//	Relative to corner:
	|	LeftTop
	|	RightTop
	|	LeftBottom
	|	RightBottom
	//	Relative in next line:
	|	Left
	|	Center
	|	Right
	//	Relative to other item:
	|	LeftOf	Id
	|	RightTo	Id
	|	Above	Id
	|	Below	Id
	//	Relative to previous item:
	|	LeftOfPrev
	|	RightToPrev
	|	AbovePrev
	|	BelowPrev
::	ItemOffset
	:==	Vector

instance	== ItemLoc						// Constructor and value equality


/*	The Direction type.
*/
::	Direction
	=	Horizontal
	|	Vertical

instance	== Direction					// Constructor equality


/*	Document interface type of interactive processes.
*/
::	DocumentInterface
	=	NDI									// No       Document Interface
	|	SDI									// Single   Document Interface
	|	MDI									// Multiple Document Interface

instance	== DocumentInterface			// Constructor equality


/*	Process attributes.
*/
::	ProcessAttribute ps						// Default:
	=	ProcessWindowPos	ItemPos			// Platform dependent
	|	ProcessWindowSize	Size			// Platform dependent
	|	ProcessWindowResize	(ProcessWindowResizeFunction ps)
											// Platform dependent
	|	ProcessHelp			(IOFunction ps)	// No Help  facility
	|	ProcessAbout		(IOFunction ps)	// No About facility
	|	ProcessActivate		(IOFunction ps)	// No action on activate
	|	ProcessDeactivate	(IOFunction ps)	// No action on deactivate
	|	ProcessClose		(IOFunction ps)	// Process is closed
	|	ProcessShareGUI						// Process does not share parent GUI
//	Attributes for MDI processes only:
	|	ProcessNoWindowMenu					// Process has WindowMenu

::	ProcessWindowResizeFunction ps
	:==	Size								// Old ProcessWindow size
	 ->	Size								// New ProcessWindow size
	 ->	ps -> ps


/*	Frequently used function types.
*/
::	IOFunction			ps	:==						ps -> ps
::	ModsIOFunction		ps	:==	Modifiers		->	ps -> ps
::	MouseFunction		ps	:== MouseState		->	ps -> ps
::	KeyboardFunction	ps	:== KeyboardState	->	ps -> ps


/*	Common error report types.
*/
::	ErrorReport								// Usual cause:
	=	NoError								// Everything went allright
	|	ErrorViolateDI						// Violation against DocumentInterface
	|	ErrorIdsInUse						// Object contains Ids that are bound
	|	ErrorUnknownObject					// Object can not be found
	|	ErrorNotifierOpen					// It was tried to open a second send notifier

instance	==			ErrorReport			// Constructor equality
instance	toString	ErrorReport			// Constructor as String

::	OkBool									// iff True, the operation was successful
	:==	Bool
